'use client';

import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { ContactFormSchema, submitContactForm } from "@/lib/apis/contact_us";
import { useState } from 'react';

export default function ContactForm() {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitSuccess, setSubmitSuccess] = useState(false);
  const [submitError, setSubmitError] = useState('');

  const {
    register,
    handleSubmit,
    reset,
    formState: { errors },
  } = useForm({
    resolver: zodResolver(ContactFormSchema),
  });

  const onSubmit = async (data: any) => {
    setIsSubmitting(true);
    setSubmitError('');
    
    try {
      const payload = {
        ...data,
        description: data.message,
      };
      delete payload.message;

      const result = await submitContactForm(payload);      
      if (result.success) {
        setSubmitSuccess(true);
         reset({name: '',email:'',subject:'',message:''})
      } else {
        setSubmitError(result.error || 'Failed to submit form');
      }
    } catch (error) {
      setSubmitError('An unexpected error occurred');
    } finally {
      setIsSubmitting(false);
      setTimeout(() => {
        setSubmitSuccess(false); 
        setSubmitError('');
      }, 2500);
    }
  };

  return (
    <div className="inner-contact-form">
      <div className="title">
        <h3>Get in touch</h3>
      </div>     
      
      <form onSubmit={handleSubmit(onSubmit)}>
        <div className="form-group">
          <div className="mb-4">
            <input 
              type="text" 
              className={`form-control ${errors.name ? 'is-invalid' : ''}`} 
              id="name" 
              placeholder="Full name" 
              {...register('name')}
            />
            {errors.name && (
              <div className="invalid-feedback">
                {errors.name.message as string}
              </div>
            )}
          </div>
          <div className="mb-4">
            <input 
              type="email" 
              className={`form-control ${errors.email ? 'is-invalid' : ''}`} 
              id="email" 
              placeholder="Email address" 
              {...register('email')}
            />
            {errors.email && (
              <div className="invalid-feedback">
                {errors.email.message as string}
              </div>
            )}
          </div>
        </div>
        <div className="form-group">
          <div className="mb-4">
            <input 
              type="text" 
              className={`form-control ${errors.subject ? 'is-invalid' : ''}`} 
              id="subject" 
              placeholder="Subject" 
              {...register('subject')}
            />
            {errors.subject && (
              <div className="invalid-feedback">
                {errors.subject.message as string}
              </div>
            )}
          </div>
        </div>
        <div className="mb-4">
          <textarea 
            className={`form-control ${errors.message ? 'is-invalid' : ''}`} 
            id="message" 
            cols={10} 
            rows={5} 
            placeholder="Write a message" 
            {...register('message')}
          />
          {errors.message && (
            <div className="invalid-feedback">
              {errors.message.message as string}
            </div>
          )}
        </div>

        {submitSuccess && (
        <div className="alert alert-success mb-4">
          Thank you for your message!
        </div>
      )}
      
      {submitError && (
        <div className="alert alert-danger mb-4">
          {submitError}
        </div>
      )}

        <button 
          type="submit" 
          className="btnprimary"
          disabled={isSubmitting}
        >
          {isSubmitting ? 'Submitting...' : 'Submit'}
          {!isSubmitting && (
            <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M5 12H19" stroke="white" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"></path>
              <path d="M12 5L19 12L12 19" stroke="white" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"></path>
            </svg>
          )}
        </button>
      </form>
    </div>
  );
}