import { useEffect, useState, useRef } from "react";
import ImageComp from "../Shared/Image";

const countries = [
  { lang: "en", label: "EN", flag: "https://flagcdn.com/w40/us.png" },
  { lang: "fr", label: "FR", flag: "https://flagcdn.com/w40/fr.png" },
  { lang: "de", label: "DE", flag: "https://flagcdn.com/w40/de.png" },
  { lang: "es", label: "ES", flag: "https://flagcdn.com/w40/es.png" },
  { lang: "ru", label: "RU", flag: "https://flagcdn.com/w40/ru.png" },
  { lang: "zh-CN", label: "ZH", flag: "https://flagcdn.com/w40/cn.png" },
];


export default function GoogleTranslateWidget() {
  const [selected, setSelected] = useState(countries[0]);
  const initialized = useRef(false);

  useEffect(() => {
    if (initialized.current) return;

    const scriptId = "google-translate-script";
    if (document.getElementById(scriptId)) {
      return;
    }

    const script = document.createElement("script");
    script.id = scriptId;
    script.src = "//translate.google.com/translate_a/element.js?cb=googleTranslateElementInit";
    script.async = true;
    document.body.appendChild(script);

    window.googleTranslateElementInit = () => {
      initialized.current = true;
      new window.google.translate.TranslateElement(
        {
          pageLanguage: "en",
          includedLanguages: "en,fr,de,es,ru,zh-CN",
          layout: window.google.translate.TranslateElement.InlineLayout.HORIZONTAL,
          autoDisplay: false,
        },
        "google_translate_element"
      );

      document.body.classList.add("translator-active");

      //useEffect(() => {
        const storedLang = localStorage.getItem("selectedLang") || "en";
      //}, []);
      if (storedLang) {
        const matchingCountry = countries.find((c) => c.lang === storedLang);
        if (matchingCountry) {
          setSelected(matchingCountry);
        }
      }
    };

    return () => {
      // Cleanup
      if (window.google && window.google.translate) {
        delete window.google.translate;
      }
      const scriptElement = document.getElementById(scriptId);
      if (scriptElement) {
        document.body.removeChild(scriptElement);
      }
    };
  }, []);

  useEffect(() => {
    if (!initialized.current) return;

    const changeLanguage = () => {
      const select = document.querySelector(".goog-te-combo");
      if (select && select.value !== selected.lang) {
        select.value = selected.lang;
        select.dispatchEvent(new Event("change"));
      }
    };

    // Try immediately and then set up a fallback interval
    changeLanguage();
    const interval = setInterval(changeLanguage, 100);
    
    return () => clearInterval(interval);
  }, [selected]);

  const handleSelect = (country) => {
    setSelected(country);
    localStorage.setItem('selectedLang', country.lang);
  };

  return (
    <>
      <div id="google_translate_element" style={{ display: 'none' }} />

      <div className="dropdown">
        <button
          className="dropdown-toggle notranslate"
          type="button"
          id="dropdownMenuButton"
          data-bs-toggle="dropdown"
          aria-expanded="false"
        >
          <ImageComp src={selected.flag} alt={selected.lang} width={24} height={17} /> {selected.label}
        </button>
        <ul className="dropdown-menu notranslate" aria-labelledby="dropdownMenuButton">
          {countries.map((country) => (
            <li key={country.lang} onClick={() => handleSelect(country)} style={{ cursor: 'pointer' }}>
              <ImageComp src={country.flag} alt={country.lang} width={24} height={17} /> {country.label}
            </li>
          ))}
        </ul>
      </div>
    </>
  );
}