import axios from 'axios';
import { z } from 'zod';

export const ContactFormSchema = z.object({
  name: z.string().trim().min(1, { message: "Full name is required" }),
  email: z.string().trim().email({ message: "Invalid email address" }),
  subject: z.string().trim().min(1, { message: "Subject is required" }),
  message: z.string().trim().min(10, { message: "Message must be at least 10 characters" }),
});

export type ContactFormValues = z.infer<typeof ContactFormSchema>;
const API_BASE_URL = process.env.NEXT_PUBLIC_API_BASE_URL || 'http://localhost:3000/api/v1';

export const submitContactForm = async (values: ContactFormValues) => {

    try {
        const response = await axios.post(`${API_BASE_URL}/get-in-touch`, values, {
            headers: {
                'Content-Type': 'application/json',
            },
        });
        return { success: true, data: response.data };
    } catch (error: any) {
        console.error('Error submitting contact form:', error);
        return {
            success: false,
            error: error.response?.data?.message || 'Failed to submit form',
        };
    }

};